/* ============================================================================
 *  Projekt:    DDS-Sinusgenerator mit ATtiny25 (PWM + DDS)
 *  Plattform:  ATtiny25 (DIP-8)
 *  IDE:        Arduino IDE 2.2.1 (mit ATtinyCore) - Programmer: USBasp (ISP, kein Bootloader)
 *
 *  Der ATtiny25 verarbeitet serielle Eingangsdaten wie ein CD4094-Schieberegisters (DATA, CLOCK, STROBE)
 *  Abhängig vom empfangenen Wert wird mittels einer Tabelle per Direct Digital Synthesis (DDS) und 256 Stufen PWM ein sinusähnliches Ausgangssignal erzeugt
 *  Frequenzbereich wie vom TF-211 RH vorgegeben: ca. 67 Hz bis 250 Hz (diskrete Stufen)
 *  Systemtakt: 8 MHz interner RC-Oszillator (kein externer Quarz). Frequenzgenauigkeit wegen RC-Oszillator ca. ±1 %
 *
 *  Pinbelegung (ATtiny25 DIP-8):
 *
 *        +------------U------------+
 *  PB5  | 1  RESET (nicht benutzt)  | 8  VCC
 *  PB3  | 2  STROBE                 | 7  PB2 CLOCK
 *  PB4  | 3  TONE_C                 | 6  PB1 PWM_OUT (OC0A)
 *  GND  | 4                         | 5  PB0 DATA
 *        +-------------------------+
 *
 * --------------------------------------------------------------------------
 *  Timer-Nutzung:
 *  Timer0:
 *    - Fast PWM
 *    - OC0A (PB1) als PWM-Ausgang
 *    - DDS-Phasenakkumulator im Timer-Interrupt
 * --------------------------------------------------------------------------
 *  Fuse-Einstellungen (Arduino IDE / TinyCore):
 *    Clock:   8 MHz Internal RC Oscillator
 *    CKDIV8:  deaktiviert
 *
 *    LFUSE = 0xE2
 *    HFUSE = 0xDF  (RESET aktiv)
 *    EFUSE = 0xFF  (BOD aus)
 * --------------------------------------------------------------------------
 *  Autor:      Armin Duft
 *  Datum:      11.01.2026
 *  Version:    1.5
 *  1.4
 *  SPI hinzu, empfängt CTCSS Frequenz Index, Lookup Tabelle für Frequenzezuordnung zum Index
 *  Aktivierung durch TONE_C Eingangssignal vom Transceiver
 *  Frequenzen passen, aber Datenübernahme bei Frequenzwechsel manchmal noch instabil
 *  1.5
 *  Tabellen zusammenführen keine Suche sondern direkt mit Index darauf zugreifen
 *  SPI mit Polling, DDS während Datenempfang abgeschaltet, dadurch Timingproblem beseitigt
 *  Mit Transceiver in Stellung "ENC" funktioniert die Signalerzeugung zuverlässig
 * ========================================================================== */

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <stdint.h>

// ------------------------- Pins -------------------------
#define DATA_PIN   PB0
#define PWM_PIN    PB1      // OC0B
#define CLOCK_PIN  PB2
#define STROBE_PIN PB3
#define TONE_C     PB4      // HIGH: Ton aktiv, LOW: Ton aus
// ------------------- Schieberegsiter ----------------------
#define LATCH_MAX 126       // gültiger Wertebereich der empfangenen Daten
#define LATCH_MIN 85
volatile uint8_t shift_reg = 0;
volatile uint8_t latch_reg = 0;
// ---------------------- DDS / PWM -------------------------
#define DDS_BITS 32
volatile uint32_t phase_accu = 0;
volatile uint32_t phase_step = 0;

// Sinus Tabelle im Flash (berechnet mit Excel)
const uint8_t sine_table[256] PROGMEM = {
127,130,133,136,140,143,146,149,152,155,158,161,164,167,170,173,
176,179,182,185,187,190,193,195,198,201,203,206,208,211,213,215,
218,220,222,224,226,228,230,232,233,235,237,238,240,241,243,244,
245,246,248,249,249,250,251,252,253,253,254,254,254,255,255,255,
255,255,255,255,254,254,254,253,253,252,251,250,249,249,248,246,
245,244,243,241,240,238,237,235,233,232,230,228,226,224,222,220,
218,215,213,211,208,206,203,201,198,195,193,190,187,185,182,179,
176,173,170,167,164,161,158,155,152,149,146,143,140,136,133,130,
127,124,121,118,114,111,108,105,102, 99, 96, 93, 90, 87, 84, 81,
 78, 75, 72, 69, 67, 64, 61, 59, 56, 53, 51, 48, 46, 43, 41, 39,
 36, 34, 32, 30, 28, 26, 24, 22, 21, 19, 17, 16, 14, 13 ,11, 10,
  9,  8,  6,  5,  5,  4,  3,  2,  1,  1,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  1,  1,  2,  3,  4,  5,  5,  6,  8,
  9, 10, 11, 13, 14, 16, 17, 19, 21, 22, 24, 26, 28, 30, 32, 34,
 36, 39, 41, 43, 46, 48, 51, 53, 56, 59, 61, 64, 67, 69, 72, 75,
 78, 81, 84, 87, 90, 93, 96, 99,102,105,108,111,114,118,121,124
};

// DDS step Tabelle 42 Frequenzen für 8 MHz Systemtakt - Latchwert --> DDS Schritt
const uint32_t dds_table[42] PROGMEM = {
  9208410,  9881861, 10582799, 11338714, 12163347, 13029213, 13743895, 14224932,
 14733456, 15241980, 15777992, 16327748, 16904991, 17495979, 18114454, 18760417,
 19420124, 20093575, 20808258, 21536684, 22292598, 23076000, 23886890, 24725268,
 25591133, 26498230, 27968827, 28958387, 29975436, 31019972, 32105740, 33232739,
 34400970,  9208410,  9881861, 10225458, 10582799, 10953885, 11338714, 11737287,
 12163347, 12575664
};

// ------------------------- Timer0 ISR (DDS) -------------------------
ISR(TIM0_OVF_vect) {
  if(PINB & (1 << TONE_C)) {                      // TONE_C Eingang direkt abfragen
    phase_accu += phase_step;
    uint8_t idex = phase_accu >> 24;              // obere 8 Bits für 256er Sinustabelle
    OCR0B = pgm_read_byte(&sine_table[idex]);
  } 
  else {
    OCR0B = 128;                                  // Ton aus: festen Mittelwert ausgeben
  }
}

// ------------------------- Schieberegister -------------------------
void update_shift(uint8_t current, uint8_t changed) {
  static uint8_t shift_reg_local = 0;
  static uint8_t dds_paused = 0;

  // --- Clock steigende Flanke ---
  if((changed & (1 << CLOCK_PIN)) && (current & (1 << CLOCK_PIN))) {
    if(!dds_paused) {                             // DDS beim ersten Datenbit pausieren
      TIMSK &= ~(1 << TOIE0);                     // Timer0 Overflow Interrupt deaktivieren
      dds_paused = 1;
    }

    shift_reg_local <<= 1;                        // Register schieben
    if(current & (1 << DATA_PIN)) {               // Daten ins Register schreiben
      shift_reg_local |= 1;
    }
  }

  // --- Strobe steigende Flanke ---
  if((changed & (1 << STROBE_PIN)) && (current & (1 << STROBE_PIN))) {   //  _|-
    latch_reg = shift_reg_local;                  // Register ins Latch übertragen
    // Bereich prüfen
    if(latch_reg >= LATCH_MIN && latch_reg <= LATCH_MAX) {
      uint8_t idx = 126 - latch_reg;
      phase_step = pgm_read_dword(&dds_table[idx]);
      phase_accu = 0;
    }
    TIMSK |= (1 << TOIE0);                        // Timer0 Overflow Interrupt wieder aktivieren
    dds_paused = 0;
  }
}

// ------------------------- Setup -------------------------
void setup() {
  DDRB &= ~((1 << DATA_PIN) | (1 << CLOCK_PIN) | (1 << STROBE_PIN) | (1 << TONE_C));  // Eingänge
  PORTB |= (1 << DATA_PIN)  | (1 << CLOCK_PIN) | (1 << STROBE_PIN) | (1 << TONE_C);   // PullUps

  DDRB  |= (1 << PWM_PIN);                                  // PB1 = OC0B Output
  TCCR0A = (1 << WGM00) | (1 << WGM01) | (1 << COM0B1);     // Fast PWM 8-bit, OC0B
  TCCR0B = (1 << CS00);                                     // Prescaler 1
  TIMSK |= (1 << TOIE0);                                    // Timer Overflow Interrupt

  sei();                                                    // Interrupts freigeben
}

// ------------------------- Loop -------------------------
void loop() {
    static uint8_t last_port = 0;

    uint8_t current_port = PINB;                            // gesamten Port einlesen
    uint8_t changed = current_port ^ last_port;             // mit XOR prüfen, ob sich etwas geändert hat
    last_port = current_port;

    if(changed) {                                           // nur bearbeiten wenn sich etwas geändert hat
        update_shift(current_port, changed);
    }
}

// -------------------- ENDE -----------------------------